import time
import re
import signal
import sys
import os
from googletrans import Translator
from colorama import init, Fore
import threading

init(autoreset=True)  # Automatically reset colors after each print

class ChatTranslator:
    DEST_LANG = "en"  # Set the destination language as a class variable

    def __init__(self, log_file_path):
        self.log_file_path = log_file_path
        self.last_message = ""
        self.translator = Translator()

    def google_translate(self, input_text):
        try:
            result = self.translator.translate(input_text, dest=self.DEST_LANG)
            return result.text, result.src
        except Exception as e:
            print(f"Translation error: {e}")
            return input_text, self.DEST_LANG

    def translate(self, input_text):
        return self.google_translate(input_text)

    def get_col_from_tag(self, tag):
        color_mapping = {
            "CT": Fore.LIGHTBLUE_EX,
            "T": Fore.YELLOW,
            "ALL": Fore.WHITE
        }
        return color_mapping.get(tag, Fore.RED)

    def parse_message(self, line):
        pattern = r'(\d{2}/\d{2} \d{2}:\d{2}:\d{2})\s+\[([A-Z]+)\]\s+(.*?)(?:﹫.*)?:\s+(.*)'
        match = re.match(pattern, line)
        if match:
            timestamp, tag, name, message = match.groups()
            return timestamp, tag, name.strip(), message.strip()
        else:
            return None, None, None, None

    def process_message(self, message):
        timestamp, tag, name, message = self.parse_message(message)
        if timestamp and tag and name and message:
            translated_message, src_lang = self.translate(message)
            
            print(f"{self.get_col_from_tag(tag)}[{tag}] {name}: {message}")
            if src_lang != self.DEST_LANG:
                print(f"{self.get_col_from_tag(tag)}[{src_lang.upper()}] -> {translated_message}")
            print()
        else:
            print('Invalid message')

    def get_last_chat(self, n=10):
        try:
            with open(self.log_file_path, encoding='utf-8', errors='replace') as f:
                lines = f.readlines()[-n:]
                lines.reverse()

            for line in lines:
                if '  [ALL] ' in line or '  [CT] ' in line or '  [T] ' in line:
                    return line.strip()
        except FileNotFoundError:
            print("Log file not found. Please check the path.")
        except Exception as e:
            print(f"Error reading log file: {e}")
        return None

    def clear_console(self):
        os.system('cls' if os.name == 'nt' else 'clear')
        print("Console cleared. Ready for new match.")

    def signal_handler(self, sig, frame):
        if sig == signal.SIGINT:
            print("\nTerminating script...")
            os._exit(0)

    def keyboard_listener(self):
        print("Type 'clear' to clear the console, 'quit' or 'exit' to exit the application.")
        while True:
            command = input().strip().lower()
            if command == 'clear':
                self.clear_console()
            elif command == 'quit' or command == 'exit':
                print("Exiting the application...")
                os._exit(0)

    def start(self):
        signal.signal(signal.SIGINT, self.signal_handler)  # Setup signal handler for Ctrl+C
        
        # Start a thread to listen for keyboard input
        threading.Thread(target=self.keyboard_listener, daemon=True).start()
        
        try:
            while True:
                cur_message = self.get_last_chat()
                if cur_message and cur_message != self.last_message:
                    self.process_message(cur_message)
                self.last_message = cur_message
                time.sleep(1)
        except KeyboardInterrupt:
            print("Terminating script...")
        finally:
            print("Cleanup and exit")

if __name__ == "__main__":
    log_file_path = r"C:\Program Files (x86)\Steam\steamapps\common\Counter-Strike Global Offensive\game\csgo\console.log"
    chat_translator = ChatTranslator(log_file_path)
    chat_translator.start()