import time
import re
import threading
from googletrans import Translator
import tkinter as tk
from tkinter import scrolledtext, messagebox, ttk

COMMON_LANGUAGES = {
    "English": "en",
    "French": "fr",
    "German": "de",
    "Spanish": "es",
    "Italian": "it",
    "Portuguese": "pt",
    "Dutch": "nl",
    "Russian": "ru",
    "Chinese": "zh-CN",
    "Japanese": "ja",
    "Korean": "ko"
}

FLAGS_EMOJI = {
    "en": "🇺🇸",
    "fr": "🇫🇷",
    "de": "🇩🇪",
    "es": "🇪🇸",
    "it": "🇮🇹",
    "pt": "🇵🇹",
    "nl": "🇳🇱",
    "ru": "🇷🇺",
    "zh-CN": "🇨🇳",
    "ja": "🇯🇵",
    "ko": "🇰🇷"
}

class ChatMonitor:
    def __init__(self, root):
        self.root = root
        self.root.title("CS:GO Chat Monitor")
        self.root.configure(bg="#333333")

        self.log_file_path = r"C:\Program Files (x86)\Steam\steamapps\common\Counter-Strike Global Offensive\game\csgo\console.log"

        self.stop_event = threading.Event()
        self.monitoring_thread = None

        self.font_size = tk.IntVar(value=18)  # Changed default font size to 18
        self.dest_lang = tk.StringVar(value="en")  # Default destination language is English

        self.create_widgets()

    def create_widgets(self):
        self.text_area = scrolledtext.ScrolledText(self.root, wrap=tk.WORD, bg="#111111", fg="#FFFFFF",
                                                    font=("Helvetica", self.font_size.get()))
        self.text_area.pack(expand=True, fill=tk.BOTH)

        self.font_size_label = ttk.Label(self.root, text="Font Size:", background="#333333", foreground="white")
        self.font_size_label.pack(side=tk.LEFT, padx=5, pady=5)

        self.font_size_spinbox = tk.Spinbox(self.root, from_=8, to=36, textvariable=self.font_size,
                                            command=self.update_font_size, width=5)
        self.font_size_spinbox.pack(side=tk.LEFT, padx=5, pady=5)

        self.dest_lang_label = ttk.Label(self.root, text="Destination Language:", background="#333333", foreground="white")
        self.dest_lang_label.pack(side=tk.LEFT, padx=5, pady=5)

        self.dest_lang_combobox = ttk.Combobox(self.root, textvariable=self.dest_lang,
                                               values=list(COMMON_LANGUAGES.keys()), state="readonly")
        self.dest_lang_combobox.pack(side=tk.LEFT, padx=5, pady=5)
        self.dest_lang_combobox.bind("<<ComboboxSelected>>", self.update_dest_lang)

        self.start_button = ttk.Button(self.root, text="Start", command=self.start_monitoring)
        self.start_button.pack(side=tk.LEFT, padx=5, pady=5)

        self.stop_button = ttk.Button(self.root, text="Stop", command=self.stop_monitoring, state=tk.DISABLED)
        self.stop_button.pack(side=tk.LEFT, padx=5, pady=5)

    def google_translate(self, input_text):
        translator = Translator()
        try:
            result = translator.translate(input_text, dest=self.dest_lang.get())
            return result.text, result.src
        except Exception as e:
            print(f"Translation error: {e}")
            return input_text, self.dest_lang.get()

    def process_message(self, message):
        timestamp, tag, name, message = self.parse_message(message)
        if timestamp and tag and name and message:
            translated_message, src_lang = self.google_translate(message)

            original_msg = f"[{tag}] {name}: {message}"
            if src_lang != self.dest_lang.get():
                translated_msg = f"[{src_lang.upper()}] -> {translated_message}"
                formatted_msg = f"{original_msg}\n{translated_msg}"
            else:
                formatted_msg = original_msg
            return f"{formatted_msg}\n{'-' * 50}"
        else:
            return 'Invalid message\n'

    def parse_message(self, line):
        pattern = r'(\d{2}/\d{2} \d{2}:\d{2}:\d{2})\s+\[([A-Z]+)\]\s+(.*?)(?:﹫.*)?:\s+(.*)'
        match = re.match(pattern, line)
        if match:
            timestamp, tag, name, message = match.groups()
            return timestamp, tag, name.strip(), message.strip()
        else:
            return None, None, None, None

    def get_last_chat(self, log_file, n=10):
        try:
            with open(log_file, encoding='utf-8', errors='replace') as f:
                lines = f.readlines()[-n:]
                lines.reverse()

            for line in lines:
                if '  [ALL] ' in line or '  [CT] ' in line or '  [T] ' in line:
                    return line.strip()
        except FileNotFoundError:
            print("Log file not found. Please check the path.")
        except Exception as e:
            print(f"Error reading log file: {e}")
        return None

    def monitor_log(self):
        last_message = ""
        while not self.stop_event.is_set():
            cur_message = self.get_last_chat(self.log_file_path)
            if cur_message and cur_message != last_message:
                formatted_message = self.process_message(cur_message)
                if formatted_message:
                    self.text_area.insert(tk.END, formatted_message + '\n')
                    self.text_area.see(tk.END)
                last_message = cur_message
            time.sleep(1)

    def start_monitoring(self):
        self.stop_event.clear()
        self.monitoring_thread = threading.Thread(target=self.monitor_log, daemon=True)
        self.monitoring_thread.start()
        self.start_button.config(state=tk.DISABLED)
        self.stop_button.config(state=tk.NORMAL)

    def stop_monitoring(self):
        self.stop_event.set()
        if self.monitoring_thread and self.monitoring_thread.is_alive():
            self.monitoring_thread.join()
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)

    def update_font_size(self):
        font_size = self.font_size.get()
        self.text_area.configure(font=("Helvetica", font_size))

    def update_dest_lang(self, event=None):
        dest_lang = COMMON_LANGUAGES[self.dest_lang_combobox.get()]
        self.dest_lang.set(dest_lang)

def main():
    root = tk.Tk()
    chat_monitor = ChatMonitor(root)
    root.mainloop()

if __name__ == "__main__":
    main()
